<?php

namespace App\Modules\Booking\Infraestructure;

use App\Modules\Booking\Application\Find\BookingNotFound;
use App\Modules\Booking\Application\Find\NotFoundBooking;
use App\Modules\Booking\Domain\BookingCode;
use App\Modules\Booking\Domain\BookingRepository;
use App\Modules\Booking\Domain\ReportExcelRequirements;
use App\Modules\Booking\Domain\UpdateManuallyConfirmedBooking;
use App\Modules\Shared\Domain\DateEnd;
use App\Modules\Shared\Domain\DateStart;
use App\Modules\Shared\Domain\Functional\Either;
use App\Modules\Shared\Domain\Storage\Criteria\StorageCriteria;
use App\Modules\Shared\Infraestructure\Requester;
use Illuminate\Support\Facades\DB;

class MysqlBookingRepository implements BookingRepository
{

    public function search(BookingCode $code)
    {
        return Requester::requestSelect(
            'CALL sp_booking_search_by_code(?,@sts,@cod,@msg)',
            [
                $code->value(),
            ]
        );
    }

    public function searchConfirmExcel(
        DateStart $dateStart,
        DateEnd $dateEnd,
        string $orderBy
    ) {
        return Requester::requestSelect(
            'CALL sp_booking_search_confirm_excel(?,?,?,@sts,@cod,@msg)',
            [
                $dateStart->value(),
                $dateEnd->value(),
                $orderBy
            ]
        );
    }

    public function searchConfirm(
        DateStart $dateStart,
        DateEnd $dateEnd,
        string $orderBy
    ) {
        return Requester::requestSelect(
            'CALL sp_booking_search_confirm(?,?,?,@sts,@cod,@msg)',
            [
                $dateStart->value(),
                $dateEnd->value(),
                $orderBy
            ]
        );
    }

    public function updateManuallyConfirmed(UpdateManuallyConfirmedBooking $model)
    {
        return Requester::requestInsert(
            'CALL sp_booking_update_manually_confirmed(?,?,@sts,@cod,@msg)',
            [
                $model->id()->value(),
                $model->manuallyConfirmed()->value()
            ]
        );
    }

    public function getReportExcel(ReportExcelRequirements $requirements)
    {
        return DB::select(
            'call sp_booking_report_excel(?,?,?,?,?,?)',
            [
                $requirements->getQuery(),
                $requirements->getDateStart(),
                $requirements->getDateEnd(),
                $requirements->getBooking_Type()->getValue(),
                $requirements->getBooking_Status()->getValue(),
                $requirements->getUser_Type()->getValue(),
            ]
        );
    }

    /**
     * @return Either<Booking, BookingNotFound>
     */
    public function find(StorageCriteria $criteria): Either
    {
        $data = $criteria->run();
        if ($data === null) {
            return Either::left(new BookingNotFound());
        }
        return Either::right($data[0]);
    }
}
