<?php

namespace App\Http\Controllers\Owner;

use App\Http\Controllers\Controller;
use App\Http\Controllers\Owner\Passenger\PassengerUpdater;
use Illuminate\Http\Request;
use App\Http\Helpers\ResponseBuilder;
use App\Http\Helpers\ParamsValidator;
use App\Http\History\HistoryData;
use App\Http\History\HistoryTables;
use App\Modules\CashMovement\Application\CashMovementUpdater;
use App\Modules\CashMovement\Infraestructure\Validators\CashMovementUpdate;
use App\Modules\CashMovement\Infraestructure\CashMovementValidators;
use App\Modules\CashMovement\Infraestructure\MysqlCashMovementUpdateRepository;
use App\Modules\Shared\Validator\Infraestructure\LaravelValidator;
use App\Modules\TypePayment\Domain\TypePaymentRequired;
use App\Providers\BookingMailSender;
use App\Providers\ExcelCashMovementLastReport\ExcelCashMovementLastReport;
use App\Providers\ExcelCashMovementReport\ExcelCashMovementReport;
use Illuminate\Support\Facades\DB;

class CashMovement extends Controller
{
    private $oDomain;
    private HistoryData $history;

    public function __construct(
        private PassengerUpdater $passengerUpdater
    ) {
        $this->oDomain = "Cash Movement";
        $this->history = new HistoryData(HistoryTables::CASH_MOVEMENT);
    }

    private function checkCard($type)
    {
        $cardTypes = [
            TypePaymentRequired::cardVisa['Id_TypePayment'],
            TypePaymentRequired::cardMastercard['Id_TypePayment'],
            TypePaymentRequired::cardAmericanExpress['Id_TypePayment'],
        ];
        return in_array($type, $cardTypes);
    }

    private function checkCash($type)
    {
        $cashTypes = [
            TypePaymentRequired::cash['Id_TypePayment'],
        ];
        return in_array($type, $cashTypes);
    }

    public function insert(Request $Request)
    {
        $oSucces = true;
        $oResponse = [];
        (new LaravelValidator)->validate($Request->all(), $this->oDomain, CashMovementValidators::FIRST_INSERT);
        $isCard = $this->checkCard($Request->input('Id_TypePayment') . '');
        $isCash = $this->checkCash($Request->input('Id_TypePayment') . '');
        if ($isCard) {
            (new LaravelValidator)->validate($Request->all(), $this->oDomain, CashMovementValidators::CARD_INSERT);
        } elseif ($isCash) {
            (new LaravelValidator)->validate($Request->all(), $this->oDomain, CashMovementValidators::CASH_INSERT);
        } else {
            (new LaravelValidator)->validate($Request->all(), $this->oDomain, CashMovementValidators::BASIC_INSERT);
        }

        $params = [
            $Request->header('Token'),
            $Request->ip()
        ];
        $data = DB::select('call sp_admin_session_get(?,?)', $params)[0];

        //  INGRESANDO DATOS
        $oParam = [
            $Request->input("CashMovement_DatePayment"),
            $Request->input("CashMovement_Amount"),
            $Request->input("CashMovement_Currency"),
            $Request->input("CashMovement_Name"),
            $Request->input("CashMovement_LastName"),
            $Request->input("CashMovement_Email"),
            $Request->input("CashMovement_Country"),
            $Request->input("CashMovement_City"),
            $Request->input("CashMovement_Address"),
            $Request->input("CashMovement_Phone"),
            $isCard ? '' : $Request->input("CashMovement_ReceiptNumber"),
            $Request->input("CashMovement_CardLast"),
            $isCard ? $Request->input("CashMovement_CardNumber") : '---',
            $Request->input("CashMovement_CardBrand"),
            $Request->input("CashMovement_CardType"),
            $Request->input("CashMovement_CardIssuer"),
            $Request->input("CashMovement_Remark"),
            $Request->input("Id_TypePayment"),
            $data->Response_Id_Admin
        ];

        $oData = DB::select('call sp_cash_movement_insert(?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)', $oParam);

        //  RESPONSE
        if ($oData[0]->Response_Success == 1) {
            $idCashMovement = $oData[0]->Id_CashMovement;
            $typePayment = DB::select('CALL sp_type_payment_index(?)', [$Request->input("Id_TypePayment")]);
            $oInvoice = json_decode($Request->input("Invoices"));
            $length = count($oInvoice);
            $idBooking = 0;
            for ($k = 0; $k < $length; $k++) {
                if ($oInvoice[$k]->Amount == 0) {
                    continue;
                }
                $oDataCashDetail = DB::select(
                    'call sp_cash_movement_detail_insert(?,?,?,?,?)',
                    [
                        $oInvoice[$k]->Amount,
                        $oData[0]->Id_CashMovement,
                        $oInvoice[$k]->Id_Invoice,
                        ($Request->input('Require_Gateway') == 2 && $typePayment[0]->TypePayment_RequireGateway == 2)
                            ? 1
                            : 0,
                        1
                    ]
                );

                if ($oDataCashDetail[0]->Response_Success == 0) {
                    $invoice = DB::select('CALL sp_invoice_get(?)', [$oInvoice[$k]->Id_Invoice]);
                    $oSucces = false;
                    break;
                } else {
                    $invoice = DB::select('CALL sp_invoice_get(?)', [$oInvoice[$k]->Id_Invoice]);
                    if (count($invoice)) {
                        $this->passengerUpdater->updateLockHtlCampBookingPassenger($invoice[0]->Id_Passenger);
                        $idBooking = $invoice[0]->Id_Booking;
                    }
                    //BookingMailSender::sendBookingByMailInvoice($oInvoice[$k]->Id_Invoice, $this->oDomain);
                }
            }

            if ($oSucces == true) {
                $oData = array(
                    array(
                        "Response_Status"   => 200,
                        "Response_Code"     => 200,
                        "Response_Message"  => "Pagos guardados con éxito",
                        "Response_Reason"   => null
                    )
                );

                $oData = json_decode(json_encode($oData));

                if ($idBooking != 0) {
                    $this->history->insert(
                        [$Request->all()],
                        $Request,
                        $idBooking
                    );
                }
            }
        } else {
        }

        $oResponse["Response_Status"]           = $oData[0]->Response_Status;
        $oResponse["Response_Code"]             = $oData[0]->Response_Code;
        $oResponse["Response_Domain"]           = $this->oDomain;
        $oResponse["Response_Message"]          = $oData[0]->Response_Message;
        $oResponse["Response_Data"]             = null;
        $oResponse["Response_Error_Message"]    = $oData[0]->Response_Message;
        $oResponse["Response_Error_Reason"]     = $oData[0]->Response_Reason;

        return ResponseBuilder::Response($oResponse);
    }

    public function update(Request $request)
    {
        (new LaravelValidator)->validate($request->all(), $this->oDomain, CashMovementUpdate::UPDATE);
        $isCard = $this->checkCard($request->input('Id_TypePayment') . '');
        $isCash = $this->checkCash($request->input('Id_TypePayment') . '');
        if ($isCard) {
            (new LaravelValidator)->validate($request->all(), $this->oDomain, CashMovementUpdate::CARD_UPDATE);
        } elseif ($isCash) {
            (new LaravelValidator)->validate($request->all(), $this->oDomain, CashMovementUpdate::CASH_UPDATE);
        } else {
            (new LaravelValidator)->validate($request->all(), $this->oDomain, CashMovementUpdate::BASIC_UPDATE);
        }

        $this->history->obtainOld($request->input('Id_CashMovement'));

        $oResponse = (new CashMovementUpdater(new MysqlCashMovementUpdateRepository()))
            ->__invoke(
                $request->input('Id_CashMovement'),
                $request->input('CashMovement_Name'),
                $request->input('CashMovement_LastName'),
                $request->input('CashMovement_Email'),
                $request->input('CashMovement_Country'),
                $request->input('CashMovement_City'),
                $request->input('CashMovement_Address'),
                $isCard ? '' : $request->input("CashMovement_ReceiptNumber"),
                $isCard ? $request->input("CashMovement_CardNumber") : '---',
                $request->input('CashMovement_CardIssuer'),
                $request->input('CashMovement_Remark'),
                $request->input('Id_TypePayment')
            );

        $this->history->update(
            $request->input('Id_CashMovement'),
            $this->history->oldValue,
            [],
            $request
        );
        return ResponseBuilder::Response($oResponse);
    }

    public function getAllFromBooking($idBooking)
    {
        // VALIDANDO DATOS
        $oResponse = ParamsValidator::Validate_Id($idBooking, $this->oDomain);
        $action = 'active';

        //  INGRESANDO DATOS

        if ($oResponse["Response_Code"] == 200) {
            $oParam = [
                $idBooking,
                $action
            ];
            $oData_Temp = DB::select('call sp_cash_movement_list_from_booking(?,?)', $oParam);

            $oResponse["Response_Status"]           = 200;
            $oResponse["Response_Code"]             = 200;
            $oResponse["Response_Domain"]           = $this->oDomain;
            $oResponse["Response_Message"]          = $this->oDomain . " list All ";
            $oResponse["Response_Data"]             = $oData_Temp;
            $oResponse["Response_Error_Message"]    = "";
            $oResponse["Response_Error_Reason"]     = "";
        }

        return ResponseBuilder::Response($oResponse);
    }

    private function validateRefund(Request $request)
    {
        (new LaravelValidator)->validate(
            $request->all(),
            $this->oDomain,
            CashMovementValidators::FIRST_INSERT_REFUND
        );

        $isCard = $this->checkCard($request->input('Id_TypePayment') . '');
        $isCash = $this->checkCash($request->input('Id_TypePayment') . '');

        if ($isCard) {
            (new LaravelValidator)->validate(
                $request->all(),
                $this->oDomain,
                CashMovementValidators::CARD_INSERT_REFUND
            );
        } elseif ($isCash) {
            (new LaravelValidator)->validate(
                $request->all(),
                $this->oDomain,
                CashMovementValidators::CASH_INSERT_REFUND
            );
        } else {
            (new LaravelValidator)->validate(
                $request->all(),
                $this->oDomain,
                CashMovementValidators::BASIC_INSERT_REFUND
            );
        }
        return [$isCard];
    }

    public function insertRefund(Request $request)
    {
        $oSucces    = true;
        $oResponse  = [];

        list($isCard) = $this->validateRefund($request);

        $oParam = [
            $request->input("CashMovement_DatePayment"),
            $request->input("CashMovement_Amount"),
            $isCard ? '' : $request->input("CashMovement_ReceiptNumber"),
            $isCard ? $request->input("CashMovement_CardNumber") : '---',
            $request->input("CashMovement_Remark"),
            $request->input("Id_TypePayment"),
            $request->header('Token')
        ];

        $oData = DB::select('call sp_cash_movement_insert_refund(?,?,?,?,?,?,?)', $oParam);

        //  RESPONSE
        if ($oData[0]->Response_Success == 1) {
            $oInvoice = json_decode($request->input("Invoices"));
            $length = count($oInvoice);
            for ($k = 0; $k < $length; $k++) {
                if ($oInvoice[$k]->Amount == 0) {
                    continue;
                }

                DB::select(
                    'call sp_cash_movement_detail_insert_refund(?,?,?,@sts,@cod,@msg,@id)',
                    [
                        $oInvoice[$k]->Amount,
                        $oData[0]->Id_CashMovement,
                        $oInvoice[$k]->Id_Invoice
                    ]
                );

                $response = DB::select(
                    'SELECT @sts AS Response_Status,
                        @cod AS Response_Code,
                        @msg AS Response_Message,
                        @id AS Response_Id'
                );


                if ($response[0]->Response_Status <> 200) {
                    $oSucces = false;
                    break;
                } else {
                    //BookingMailSender::sendBookingByMailInvoice($oInvoice[$k]->Id_Invoice, $this->oDomain);
                }
            }

            if ($oSucces == true) {
                $oData = [
                    (object)[
                        "Response_Status"   => 200,
                        "Response_Code"     => 200,
                        "Response_Message"  => "Pagos guardados con éxito",
                        "Response_Reason"   => null
                    ]
                ];
            }
        } else {
        }

        $oResponse["Response_Status"]           = $oData[0]->Response_Status;
        $oResponse["Response_Code"]             = $oData[0]->Response_Code;
        $oResponse["Response_Domain"]           = $this->oDomain;
        $oResponse["Response_Message"]          = $oData[0]->Response_Message;
        $oResponse["Response_Data"]             = null;
        $oResponse["Response_Error_Message"]    = $oData[0]->Response_Message;
        $oResponse["Response_Error_Reason"]     = $oData[0]->Response_Reason;

        return ResponseBuilder::Response($oResponse);
    }

    public function insertDiscount(Request $Request)
    {
        $oSucces    = true;
        $oResponse  = array();
        $oValParams = array(
            'Id_InvoiceFee'                 => 'required|int',
            'CashMovement_DatePayment'      => 'required|string',
            'CashMovement_ReceiptNumber'    => 'required|string|max:100',
            'CashMovement_Remark'           => 'string',
            'Id_TypePayment'                => 'required|int'
        );


        // VALIDANDO DATOS
        $oResponse = ParamsValidator::Validate_Request($Request, $oValParams, $this->oDomain);

        //  INGRESANDO DATOS
        if ($oResponse["Response_Code"] == 200) {
            $oParam = array(
                $Request->input("Id_InvoiceFee"),
                $Request->header("Token"),
                $Request->input("CashMovement_DatePayment"),
                $Request->input("CashMovement_ReceiptNumber"),
                $Request->input("CashMovement_Remark"),
                $Request->input("Id_TypePayment")
            );

            $oData = DB::select('call sp_cash_movement_insert_discount(?,?,?,?,?,?)', $oParam);

            //  RESPONSE
            if ($oData[0]->Response_Success == 1) {
            } else {
            }

            $oResponse["Response_Status"]           = $oData[0]->Response_Status;
            $oResponse["Response_Code"]             = $oData[0]->Response_Code;
            $oResponse["Response_Domain"]           = $this->oDomain;
            $oResponse["Response_Message"]          = $oData[0]->Response_Message;
            $oResponse["Response_Data"]             = null;
            $oResponse["Response_Error_Message"]    = $oData[0]->Response_Message;
            $oResponse["Response_Error_Reason"]     = $oData[0]->Response_Reason;
        }

        return ResponseBuilder::Response($oResponse);
    }

    public function index(Request $Request, $Id)
    {
        $oResponse  = array();

        // VALIDANDO DATOS
        $oResponse = ParamsValidator::Validate_Id($Id, $this->oDomain);

        //  INGRESANDO DATOS
        if ($oResponse["Response_Code"] == 200) {
            $oParam = array($Id);
            $oData  = DB::select('call sp_cash_movement_index(?)', $oParam);

            $oResponse["Response_Status"]           = 200;
            $oResponse["Response_Code"]             = 200;
            $oResponse["Response_Domain"]           = $this->oDomain;
            $oResponse["Response_Message"]          = $this->oDomain . " Index";
            $oResponse["Response_Data"]             = $oData;
            $oResponse["Response_Error_Message"]    = "";
            $oResponse["Response_Error_Reason"]     = "";
        }

        return ResponseBuilder::Response($oResponse);
    }

    public function list(Request $Request, $Id_Invoice)
    {
        $oResponse  = array();

        // VALIDANDO DATOS
        $oResponse = ParamsValidator::Validate_Id($Id_Invoice, $this->oDomain);
        $action = 'active';

        //  INGRESANDO DATOS
        if ($oResponse["Response_Code"] == 200) {
            $oParam = [
                $Id_Invoice,
                config('var.PATH_PUBLIC'),
                $action
            ];
            $oData  = DB::select('call sp_cash_movement_list(?,?,?)', $oParam);

            $oResponse["Response_Status"]           = 200;
            $oResponse["Response_Code"]             = 200;
            $oResponse["Response_Domain"]           = $this->oDomain;
            $oResponse["Response_Message"]          = $this->oDomain . " Index";
            $oResponse["Response_Data"]             = $oData;
            $oResponse["Response_Error_Message"]    = "";
            $oResponse["Response_Error_Reason"]     = "";
        }

        return ResponseBuilder::Response($oResponse);
    }

    public function reportExcel(Request $Request)
    {
        $oResponse  = array();
        $oValParams = array(
            'Date_Start'        => 'required|string',
            'Date_End'          => 'required|string',
            'Text'              => 'string',
            'Id_TypePayment'    => 'required|numeric',
            'User_Type'         => 'required|numeric'
        );


        // VALIDANDO DATOS
        $oResponse = ParamsValidator::Validate_Request($Request, $oValParams, $this->oDomain);

        //  INGRESANDO DATOS
        if ($oResponse["Response_Code"] == 200) {
            $oParam =   [
                $Request->input("Date_Start"),
                $Request->input("Date_End"),
                $Request->input("Text"),
                $Request->input("Id_TypePayment"),
                $Request->input("User_Type"),
                config('var.PATH_PUBLIC')
            ];

            $oData = self::mapCashMovement(DB::select('call sp_cash_movement_search(?,?,?,?,?,?)', $oParam));
            (new ExcelCashMovementReport())->exportToExcel($oData);
        } else {
            return ResponseBuilder::Response($oResponse);
        }
    }

    public function lastReportExcel(Request $Request)
    {
        $oResponse  = array();
        $oValParams = array(
            'Date_Start'        => 'required|string',
            'Date_End'          => 'required|string',
            'Text'              => 'string',
            'Id_TypePayment'    => 'required|numeric',
            'User_Type'         => 'required|numeric'
        );


        // VALIDANDO DATOS
        $oResponse = ParamsValidator::Validate_Request($Request, $oValParams, $this->oDomain);

        //  INGRESANDO DATOS
        if ($oResponse["Response_Code"] == 200) {
            $oParam =   [
                $Request->input("Date_Start"),
                $Request->input("Date_End"),
                $Request->input("Text"),
                $Request->input("Id_TypePayment"),
                $Request->input("User_Type"),
                config('var.PATH_PUBLIC')
            ];

            $oData = self::mapCashMovement(DB::select('call sp_cash_movement_search(?,?,?,?,?,?)', $oParam));
            (new ExcelCashMovementLastReport())->exportToExcel($oData);
        } else {
            return ResponseBuilder::Response($oResponse);
        }
    }

    public function reportExcelJustOne(Request $Request, $Id)
    {
        $oResponse  = array();

        // VALIDANDO DATOS
        $oResponse = ParamsValidator::Validate_Id($Id, $this->oDomain);

        //  INGRESANDO DATOS
        if ($oResponse["Response_Code"] == 200) {
            $oParam =   [$Id];

            $oData = self::mapCashMovement(DB::select('call sp_cash_movement_index(?)', $oParam));

            (new ExcelCashMovementReport())->exportToExcel($oData);
        } else {
            return ResponseBuilder::Response($oResponse);
        }
    }

    public static function mapCashMovement($cashMovementList)
    {
        return array_map(
            function ($data) {
                $data->CashMovementDetail = array_map(
                    function ($cmd) {
                        $cmd->Invoice_Detail = DB::select('call sp_invoice_detail_list(?)', [$cmd->Id_Invoice]);
                        $cmd->Booking_Tour = DB::select('call sp_passenger_list_booking_tour(?)', [$cmd->Id_Passenger]);
                        $cmd->invoice =  c_Invoice::GetAll(json_encode(BookingMailSender::GetInvoiceList($cmd->Id_Booking)));
                        return $cmd;
                    },
                    DB::select('call sp_cash_movement_detail_list(?,?)', [$data->Id_CashMovement, 'active'])
                );
                return $data;
            },
            $cashMovementList
        );
    }

    public function search(Request $Request)
    {
        $oResponse  = array();
        $oValParams = array(
            'Date_Start'    => 'required|string',
            'Date_End'      => 'required|string',
            'Text'          => 'string',
            'Id_TypePayment' => 'required|numeric',
            'User_Type'         => 'required|numeric'
        );
        // VALIDANDO DATOS
        $oResponse = ParamsValidator::Validate_Request($Request, $oValParams, $this->oDomain);

        //  INGRESANDO DATOS
        if ($oResponse["Response_Code"] == 200) {
            $oParam = [
                $Request->input("Date_Start"),
                $Request->input("Date_End"),
                $Request->input("Text"),
                $Request->input("Id_TypePayment"),
                $Request->input("User_Type"),
                config('var.PATH_PUBLIC')
            ];

            $oData  = DB::select('call sp_cash_movement_search(?,?,?,?,?,?)', $oParam);

            $oResponse["Response_Status"]           = 200;
            $oResponse["Response_Code"]             = 200;
            $oResponse["Response_Domain"]           = $this->oDomain;
            $oResponse["Response_Message"]          = $this->oDomain . " Search";
            $oResponse["Response_Data"]             = $oData;
            $oResponse["Response_Error_Message"]    = "";
            $oResponse["Response_Error_Reason"]     = "";
        }

        return ResponseBuilder::Response($oResponse);
    }

    public function selector(Request $Request, $Id)
    {
        if (is_numeric($Id)) {
            return $this->index($Request, $Id);
        } elseif ($Id == 'search') {
            return $this->search($Request);
        } elseif ($Id == 'search') {
            return $this->search($Request);
        }
    }
}
