<?php

namespace App\Modules\Media\Infrastructure;

use App\Modules\Media\Domain\ImageThumbnailCreator;
use App\Modules\Shared\Domain\StringUtils;
use Intervention\Image\ImageManagerStatic;
use Imagick;

class MemoryImageThumbnailCreator implements ImageThumbnailCreator
{
    private function fitImage(string $path, string $ext, int $width, int $height)
    {
        $imageThumb = ImageManagerStatic::make($path);
        $imageThumb->fit($width, $height, function ($constraint) {
            $constraint->upsize();
        });
        return $imageThumb->encode($ext);
    }

    private function scaleImage(string $path, string $ext, int $width, int $height)
    {
        $imageThumb = ImageManagerStatic::make($path);
        $imageThumb->resize($width, $height, function ($constraint) {
            $constraint->aspectRatio();
            $constraint->upsize();
        });
        return $imageThumb->encode($ext);
    }

    public function create(string $src, int $width, int $height, string $ext): string
    {
        try {
            $info = pathinfo($src);
            $fileExtension = $info['extension'];
            if ($fileExtension === 'HEIC') {
                $path = $this->generateImageFromHeic($src);
                $encoded = $this->fitImage($path, $ext, $width, $height);
                unlink($path);
                return $encoded;
            }
            return $this->fitImage($src, $ext, $width, $height);
        } catch (\Throwable $th) {
            if (StringUtils::startsBy($th->getMessage(), 'Unsupported image type')) {
                return '';
            }
            throw $th;
        }
    }

    public function createScale(string $src, int $width, int $height, string $ext): string
    {
        try {
            $info = pathinfo($src);
            $fileExtension = $info['extension'];
            if ($fileExtension === 'HEIC') {
                $path = $this->generateImageFromHeic($src);
                $encoded = $this->scaleImage($path, $ext, $width, $height);
                unlink($path);
                return $encoded;
            }
            return $this->scaleImage($src, $ext, $width, $height);
        } catch (\Throwable $th) {
            if (StringUtils::startsBy($th->getMessage(), 'Unsupported image type')) {
                return '';
            }
            throw $th;
        }
    }

    private function generateImageFromHeic($src)
    {
        $extension = 'jpeg';
        $uploadedImage = fopen($src, 'rb');
        $image = new Imagick();
        $image->readImageFile($uploadedImage);
        $image->setFormat($extension);
        $image->setImageCompressionQuality(100);
        $path = config('var.PATH_PRIVATE') . 'temp' . DIRECTORY_SEPARATOR . uniqid() . '.' . $extension;
        $image->writeImage($path);
        fclose($uploadedImage);
        return $path;
    }
}
