<?php

namespace App\Modules\Image\Infraestructure;

use App\Modules\Image\Domain\BasicImage;
use App\Modules\Shared\Domain\IsNotAFile;
use Illuminate\Http\Request;
use Symfony\Component\HttpKernel\Exception\HttpException;
use Intervention\Image\ImageManagerStatic;
use Throwable;

class BasicImageStorage
{

    private const EXTENSION = 'png';
    private static $extension = ['jpg', 'jpeg', 'png'];

    private Request $request;

    public function __construct(Request $request)
    {
        $this->request = $request;
    }

    public function storeImage(string $inputName, $thumbSettings)
    {
        if (!$this->request->hasFile($inputName)) {
            throw new IsNotAFile($inputName);
        }
        $thumbUrl = '';
        $imageUrl = '';
        $tempUrl = '';
        try {
            $path = config('var.PATH_PRIVATE');
            $imageDir = $this->createDirIfNotExists('images', 'img');
            $thumbDir = $this->createDirIfNotExists('thumbnail', 'img');
            $ext = $this->request->file($inputName)->getClientOriginalExtension();

            if (!in_array($ext, self::$extension)) {
                throw new HttpException(406, 'El archivo no tiene una extensión válida.');
            }

            $tempUrl = $this->moveToTemp($inputName, $ext);

            $thumbUrl = $this->generateThumb($path . $thumbDir, $ext, $thumbSettings, $tempUrl);
            $imageUrl = $this->generateImage($path . $imageDir, $ext, $tempUrl);
            unlink($tempUrl);
            $tempUrl = '';
            return new BasicImage(
                $thumbDir . '/' . $thumbUrl,
                $imageDir . '/' . $imageUrl,
                self::EXTENSION,
                filesize($path . $imageDir . '/' . $imageUrl)
            );
        } catch (Throwable $ex) {
            if ($tempUrl) {
                unlink($tempUrl);
            }
            if ($thumbUrl) {
                unlink($path . $thumbDir . '/' . $thumbUrl);
            }
            if ($imageUrl) {
                unlink($path . $imageDir . '/' . $imageUrl);
            }
            throw $ex;
        }
    }

    private function generateThumb($thumbDir, $ext, $thumbSettings, $temp)
    {
        $imageThumb = ImageManagerStatic::make($temp);
        $imageThumb->resize($thumbSettings['Thumbnail.Width'], $thumbSettings['Thumbnail.Height']);
        list($thumbUrl, $thumbFile) = $this->generateFilePointer($thumbDir, $ext);
        fwrite($thumbFile, $imageThumb->encode(self::EXTENSION));
        fclose($thumbFile);
        return $thumbUrl;
    }

    private function generateImage($imageDir, $ext, $temp): string
    {
        $imageThumb = ImageManagerStatic::make($temp);
        list($imageUrl, $imageFile) = $this->generateFilePointer($imageDir, $ext);
        fwrite($imageFile, $imageThumb->encode(self::EXTENSION));
        fclose($imageFile);
        return $imageUrl;
    }

    private function generateFilePointer($dir, $ext)
    {
        $flag = 0;
        $attempts = 10;
        do {
            $generated = date('His') . uniqid() . '.' . $ext;
            $generatedUrl = $dir . '/' . $generated;
            $flag++;
            $file = false;
            if ($flag < $attempts) {
                try {
                    $file = fopen($generatedUrl, 'x');
                } catch (\Throwable $th) {
                }
            }
        } while ($file === false && $flag < $attempts);
        if ($flag >= $attempts) {
            throw new HttpException(406, 'Error generating file.');
        }
        return [$generated, $file];
    }

    private function moveToTemp($inputName, $ext): string
    {
        $tempName = uniqid() . $ext;
        $path = config('var.PATH_PRIVATE') . 'temp/';
        if (!$this->request->file($inputName)->move($path, $tempName)) {
            throw new HttpException(406, 'Error uploading file');
        }
        return $path . $tempName;
    }

    public function createDirIfNotExists($path, $prefix): string
    {
        $filesPath = config('var.PATH_PRIVATE');
        $directory = $path . '/' . $prefix . date('ym');
        $dirPath = $filesPath . $directory;

        if (!is_dir($dirPath)) {
            @mkdir($dirPath, 0700);
        }
        return $directory;
    }
}
