<?php

namespace App\Providers\UserBooking;

use App\Http\Constants\BookingConstants;
use App\Http\Controllers\Mail\c_Mail_Body;
use App\Http\Controllers\Mail\c_Mail_Structure;
use App\Http\Controllers\Mail\c_Mail_Term;
use App\Http\Controllers\Mail\c_Mail_Table;
use App\Http\Helpers\CashMovementHelper;
use App\Http\Helpers\DateTime;
use App\Providers\Translation\Expressions;

class UserBookingPdf
{
    public static $expression = Expressions::CMS['pdf.booking-confirmation'];
    public const PADDING = '0px 5px 2px 5px';

    public function build($Content)
    {
        $booking = $Content['Booking'];
        $oMail_Body = new c_Mail_Body;
        $oMail_Structure = new c_Mail_Structure;
        $oMail_Term = new c_Mail_Term;
        $booking->Tour = array_values(array_filter($booking->Tour, fn($t) => $t->BookingTour_Status == 2));

        function expressionLanguage($expression, $language)
        {
            return function ($index) use ($expression, $language) {
                return $expression[$index][$language];
            };
        }

        function createTourTable($booking)
        {
            $text = expressionLanguage(UserBookingPdf::$expression, $booking->Id_GuideLanguage);
            $tourTable = c_Mail_Table::table();

            $rows = [];
            $rows = array_merge($rows, mapTourList($booking, $booking->Tour));
            $rows = array_merge($rows, mapFlightList($booking->Flight));

            addToTable([[$text('titleTour'), $text('titleDepartureDate'), $text('titleTourBriefing')]], $tourTable, 'dark');
            addToTable($rows, $tourTable, 'base');
            return $tourTable;
        }

        function mapTourList($booking, $tourList)
        {
            return array_map(function ($tour) use ($booking) {
                $briefing = '';
                if ($tour->BookingTour_UseBookingPlaceBriefing == 1) {
                    $briefing = DateTime::dateTimeTransform($tour->BookingTour_DateBriefing) . ' ' . '(' . $tour->BookingTour_PlaceBriefing . ')';
                } else {
                    $briefing = DateTime::dateTimeTransform($booking->Booking_DateBriefing) . '(' . $booking->Booking_PlaceBriefing . ')';
                }
                return [
                    $tour->BookingTour_TourName,
                    DateTime::dateTransform($tour->BookingTour_DateStart),
                    $briefing,
                ];
            }, $tourList);
        }

        function mapFlightList($flightList)
        {
            return array_map(function ($flight) {
                $cell1 = $flight->Flight_Type == 2 ? 'TRANSFER OUT ' : 'TRANSFER IN';
                $cell1 .= "$flight->Flight_Carrier $flight->Flight_Number ($flight->Flight_Origin-$flight->Flight_Destination)";
                return [
                    $cell1,
                    DateTime::dateTimeTransform($flight->Flight_DateArrival),
                    '',
                ];
            }, $flightList);
        }

        function createUserTable($booking)
        {
            $text = expressionLanguage(UserBookingPdf::$expression, $booking->Id_GuideLanguage);

            $userTable = c_Mail_Table::table();

            $userTable
                ->basicRow([
                    c_Mail_Table::orangeColumn($text('infoContactName'))->setFontSize('9px')->setPadding(UserBookingPdf::PADDING),
                    c_Mail_Table::baseColumn("$booking->User_Name $booking->User_LastName")->setFontSize('9px')->setPadding(UserBookingPdf::PADDING),
                    c_Mail_Table::orangeColumn($text('infoWhatsapp'))->setFontSize('9px')->setPadding(UserBookingPdf::PADDING),
                    c_Mail_Table::baseColumn("+$booking->User_PhoneCode $booking->User_Phone")->setFontSize('9px')->setPadding(UserBookingPdf::PADDING),
                ])
                ->basicRow([
                    c_Mail_Table::orangeColumn($text('infoNumberOfPax'))->setFontSize('9px')->setPadding(UserBookingPdf::PADDING),
                    c_Mail_Table::baseColumn(count($booking->Passengers))->setFontSize('9px')->setPadding(UserBookingPdf::PADDING),
                    c_Mail_Table::orangeColumn($text('infoLanguage'))->setFontSize('9px')->setPadding(UserBookingPdf::PADDING),
                    c_Mail_Table::baseColumn($booking->GuideLanguage_Name)->setFontSize('9px')->setPadding(UserBookingPdf::PADDING),
                ])
                ->basicRow([
                    c_Mail_Table::orangeColumn($text('infoEmail'))->setFontSize('9px')->setPadding(UserBookingPdf::PADDING),
                    c_Mail_Table::baseColumn($booking->User_Email)
                        ->setColspan(3)
                        ->setFontSize('9px')
                        ->setPadding(UserBookingPdf::PADDING)
                ])
                ->basicRow([
                    c_Mail_Table::orangeColumn($text('infoHotelInCusco'))->setFontSize('9px')->setPadding(UserBookingPdf::PADDING),
                    c_Mail_Table::baseColumn($booking->Booking_Hotel)
                        ->setFontSize('9px')
                        ->setColspan(3)
                        ->setPadding(UserBookingPdf::PADDING),
                ])
                ->basicRow([
                    c_Mail_Table::orangeColumn($text('infoRoomType'))->setFontSize('9px')->setPadding(UserBookingPdf::PADDING),
                    c_Mail_Table::baseColumn(getRoomsResume($booking->Room))
                        ->setFontSize('9px')
                        ->setColspan(3)
                        ->setPadding(UserBookingPdf::PADDING),
                ]);
            return $userTable;
        }

        function createPaxTable($paxList, $booking)
        {
            $text = expressionLanguage(UserBookingPdf::$expression, $booking->Id_GuideLanguage);
            $paxTable = c_Mail_Table::table();

            $paxTable->basicRow([
                c_Mail_Table::baseColumn($text('paxInformation'))->setAlign('center')->setColspan(7)->setBold()
                    ->setFontSize('9px')
                    ->setPadding(UserBookingPdf::PADDING)
            ]);
            addToTable([[$text('paxName'), $text('paxSurname'), $text('paxMaleFemale'), $text('paxDob'), $text('paxAge'), $text('paxDocument'), $text('paxCountry')]], $paxTable, 'dark');
            addToTable(mapPaxList($paxList), $paxTable);
            return $paxTable;
        }

        function createTourInformationTable($invoiceDetailAll, $booking)
        {
            $text = expressionLanguage(UserBookingPdf::$expression, $booking->Id_GuideLanguage);
            $paxTable = c_Mail_Table::table();

            $paxTable->basicRow([
                c_Mail_Table::baseColumn($text('tourInformation'))->setAlign('center')->setColspan(6)->setBold()
                    ->setFontSize('9px')
                    ->setPadding(UserBookingPdf::PADDING)
            ]);
            addToTable([[$text('tourStart'), $text('tourPackage'), $text('tourInc'), $text('tourUnitPrice'), $text('tourQty'), $text('tourTotal')]], $paxTable, 'orange');

            $rows = [];
            $invoiceTotal = 0;

            foreach ($invoiceDetailAll as $detail) {
                if ($detail->InvoiceDetail_Type == BookingConstants::InvoiceDetail_TypePackage) {
                    $detail->datePackage = searchMinTourDate($detail->InvoiceDetail_Group, $booking);
                } else {
                    $detail->datePackage = '';
                }

                if ($booking->User_Type == BookingConstants::User_TypeEndorser && $detail->InvoiceDetail_Type == BookingConstants::InvoiceDetail_TypeIGV) {
                    $invoiceTotal += $detail->InvoiceDetail_Total;
                } else {
                    $rows[] = [
                        !$detail->datePackage ? '' : DateTime::dateTransform($detail->datePackage),
                        $detail->InvoiceDetail_Description,
                        $detail->InvoiceDetail_Included == 1 ? "" : $detail->InvoiceDetail_Quantity,
                        $detail->InvoiceDetail_Type == BookingConstants::InvoiceDetail_TypeIGV ? "" : (round($detail->InvoiceDetail_Price * 100)) / 100,
                        $detail->InvoiceDetail_Type == BookingConstants::InvoiceDetail_TypeIGV ? "" : $detail->InvoiceDetail_Quantity,
                        round($detail->InvoiceDetail_Total * 100) / 100,
                    ];
                }
            }

            addToTable($rows, $paxTable);
            $paxTable->basicRow([c_Mail_Table::baseColumn('<br />')->setAlign('center')->setColspan(6)->setPadding(UserBookingPdf::PADDING)]);

            $texttot = $text('infoTotal');
            $valtot = $booking->InvoiceRes['Total'];
            $rows = 1;
            $obser = '';

            if ($booking->User_Type == 2) {
                $texttot = $text('infoToursBalance');
                $valtot -= $invoiceTotal;
                $rows = 5;
                $obser = $text('infoRemark') . ": " . $booking->Booking_Observation;
            }
            $paxTable->basicRow([
                c_Mail_Table::baseColumn($obser)->setColspan(3)->setRowspan($booking->User_Type == 2 ? 3 : 1)
                    ->setFontSize('9px')->setPadding(UserBookingPdf::PADDING),
                c_Mail_Table::orangeColumn($texttot)->setColspan(2)->setBold()
                    ->setFontSize('9px')->setPadding(UserBookingPdf::PADDING),
                c_Mail_Table::baseColumn(round($valtot * 100) / 100)->setAlign('right')
                    ->setFontSize('9px')->setPadding(UserBookingPdf::PADDING)
            ]);

            if ($booking->User_Type == 2) {
                $sumpays = 0;
                foreach ($booking->PayAll as $pay) {
                    $sumpays += $pay->CashMovementDetail_Amount;
                }

                $paxTable->basicRow([
                    c_Mail_Table::orangeColumn($text('infoIGV'))->setColspan(2)->setBold()->setFontSize('9px')->setPadding(UserBookingPdf::PADDING),
                    c_Mail_Table::baseColumn(round($invoiceTotal * 100) / 100)->setAlign('right')->setFontSize('9px')->setPadding(UserBookingPdf::PADDING)
                ]);
                $paxTable->basicRow([
                    c_Mail_Table::orangeColumn($text('infoTotal'))->setColspan(2)->setBold()->setFontSize('9px')->setPadding(UserBookingPdf::PADDING),
                    c_Mail_Table::baseColumn(round(($valtot + $invoiceTotal) * 100) / 100)->setAlign('right')->setFontSize('9px')->setPadding(UserBookingPdf::PADDING)
                ]);
            }

            foreach ($booking->PayAll as $pay) {
                $paxTable->basicRow([
                    c_Mail_Table::baseColumn(
                        "$pay->TypePayment_Name: "
                            . CashMovementHelper::changeReceiptNumber($pay) . " ($pay->CashMovement_Amount )"
                    )->setColspan(3)->setFontSize('9px')->setPadding(UserBookingPdf::PADDING),
                    c_Mail_Table::orangeColumn($text('infoDeposit'))->setColSpan(2)->setBold()->setFontSize('9px')->setPadding(UserBookingPdf::PADDING),
                    c_Mail_Table::baseColumn(round($pay->CashMovementDetail_Amount * 100) / 100)->setAlign('right')->setFontSize('9px')->setPadding(UserBookingPdf::PADDING)
                ]);
            }

            $paxTable->basicRow([
                c_Mail_Table::baseColumn('')->setColspan(3)->setFontSize('9px')->setPadding(UserBookingPdf::PADDING),
                c_Mail_Table::darkColumn($text('infoBalance'))->setColSpan(2)->setBold()->setFontSize('9px')->setPadding(UserBookingPdf::PADDING),
                c_Mail_Table::baseColumn(round($booking->InvoiceRes['TotalPending'] * 100) / 100)->setAlign('right')->setFontSize('9px')->setPadding(UserBookingPdf::PADDING)
            ]);

            if ($booking->User_Type != 2) {

                $paxTable->basicRow([
                    c_Mail_Table::orangeColumn($booking->Admin_Abrv)->setColspan(6)->setAlign('center')->setBold()->setFontSize('9px')->setPadding(UserBookingPdf::PADDING)
                ]);
            }
            return $paxTable;
        }

        function searchMinTourDate($group, $booking)
        {
            return array_reduce($booking->Tour, function ($prev, $tour) use ($group) {
                if ($tour->BookingTour_Group != $group) {
                    return $prev;
                }
                if ($prev == null) {
                    return $tour->BookingTour_DateStart;
                }

                return (strtotime($prev) < strtotime($tour->BookingTour_DateStart)) ? $prev : $tour->BookingTour_DateStart;
            }, null);
        }

        function getRoomsResume($roomList)
        {
            $roomAmounts = (object) [];
            $text = '';

            foreach ($roomList as $i => $room) {
                $key = $room->Id_TypeRoom;
                if (isset($roomAmounts->$key)) {
                    $roomAmounts->$key->count++;
                } else {
                    $roomAmounts->$key = (object)[
                        'Id_TypeRoom'   => $room->Id_TypeRoom,
                        'index'     => $i,
                        'count'     => 1
                    ];
                }
            }
            $i = 0;
            foreach ($roomAmounts as $key => $amount) {
                $text .= ($i == 0 ? '' : ', ') . str_pad($amount->count, 2, '0', STR_PAD_LEFT) . ' ' . $roomList[$amount->index]->TypeRoom_Name;
                $i++;
            }
            return $text;
        }

        function bookingSession($booking)
        {
            return DateTime::dateTimeTransform($booking->Booking_DateBriefing) . '(' . $booking->Booking_PlaceBriefing . ')';
        }

        function mapPaxList($paxList)
        {
            return array_map(function ($pax) {
                return [
                    $pax->Passenger_Name,
                    $pax->Passenger_LastName,
                    $pax->Passenger_Gender == '1' ? 'M' : 'F',
                    DateTime::dateTransform($pax->Passenger_DOB),
                    $pax->Passenger_Age,
                    $pax->Passenger_NoDocument,
                    $pax->UserCountry_Name
                ];
            }, array_filter($paxList, fn ($pax) => $pax->Passenger_Status > 1));
        }

        function addToTable($rows, &$table, $type = 'base')
        {
            foreach ($rows as $row) {
                $tableRow = c_Mail_Table::row();
                $table->addRow($tableRow);
                foreach ($row as $col) {
                    switch ($type) {
                    case 'base':
                        $tableRow->addColumn(c_Mail_Table::baseColumn($col)->setFontSize('9px')->setPadding(UserBookingPdf::PADDING));
                        break;
                    case 'dark':
                        $tableRow->addColumn(c_Mail_Table::darkColumn($col)->setFontSize('9px')->setPadding(UserBookingPdf::PADDING));
                        break;
                    case 'orange':
                        $tableRow->addColumn(c_Mail_Table::orangeColumn($col)->setFontSize('9px')->setPadding(UserBookingPdf::PADDING));
                        break;
                    }
                }
            }
        }

        $HeadFunction = function ($head) use ($oMail_Body) {
            return $oMail_Body->Get_Head($oMail_Body->Get_H4_2($head));
        };
        $CellFunction = function ($cell) use ($oMail_Body) {
            return $oMail_Body->Get_Cell($oMail_Body->Get_P($cell, 'LEFT'));
        };

        $PrintObligatory = function ($fee) {
            return $fee->InvoiceFee_Full == 2 && $fee->InvoiceFee_Amount - $fee->InvoiceFee_AmountPay > 0;
        };

        

        //
        //  OBTENEMOS TERMINOS EN UN IDIOMA
        //
        $Term           = $oMail_Term->Get_Term($Content["Lang"]);

        $Title          = $Term["Booking_Title"];
        $Message        = $Term["Booking_Message"];
        $Signature      = $Term["Signature"];

        $reduce = 0.7;

        $Mail_Body      = $oMail_Body->Get_Booking_Body($reduce);
        $Mail_Content   = $oMail_Body->Get_H1($Title, $reduce);
        $Mail_Content   .= $oMail_Body->Get_P($Message, "LEFT");

        $urlType = $Content['ResourcesUrl'];

        $assets = BookingTypeImage::getAssetValue($booking);
        $urlType = $assets['asset'];

        $Mail_Content   .= $oMail_Structure->Get_Subtitle('<img style="height:14px;" src="' . $urlType . '" /> BOOKING CONFIRMATION - ' . $booking->Booking_Code, 'LEFT');
        $Mail_Content   .= createTourTable($booking);
        $Mail_Content   .= $oMail_Body->Get_P(' ', 'LEFT');
        $Mail_Content   .= createUserTable($booking);
        $Mail_Content   .= $oMail_Body->Get_P(' ', 'LEFT');
        $Mail_Content   .= createPaxTable($booking->Passengers, $booking);
        $Mail_Content   .= $oMail_Body->Get_P(' ', 'LEFT');
        $Mail_Content   .= createTourInformationTable($booking->InvoiceDetailAll, $booking);
        $Mail_Content   .= $oMail_Body->Get_P($Signature, "RIGHT");
        $Mail_Body      = str_replace("[[CONTENT]]", $Mail_Content, $Mail_Body);
        $Mail_Body      = str_replace("[[MEDIA_HEADER]]", $Content["ResourcesUrl"] . 'mail/Mail_Header.png', $Mail_Body);
        return $Mail_Body;
    }
}
