<?php

namespace App\Http\Controllers\Owner\GroupProvider;

use App\Http\Controllers\Controller;
use App\Http\Helpers\ResponseBuilder;
use App\Http\History\HistoryData;
use App\Http\History\HistoryTables;
use App\Modules\GroupProvider\Application\DownloadMedia\DownloadMediaGroupProviderQuery;
use App\Modules\GroupProvider\Application\DownloadMedia\DownloadMediaGroupProviderResponse;
use App\Modules\Shared\Domain\Bus\Query\QueryBus;
use App\Modules\Shared\Domain\Transaction;
use App\Providers\StorageProvider;
use App\Providers\ZipGenerator\ZipGenerator;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

final class GroupProviderDownloadMediaController extends Controller
{
    private string $domain = 'GroupProvider';
    private HistoryData $history;

    public function __construct(
        private QueryBus $queryBus,
        private Transaction $transaction,
        private ZipGenerator $zipGenerator,
        private StorageProvider $storageProvider
    ) {
        $this->history = new HistoryData(HistoryTables::GROUP_PROVIDER);
    }

    public function __invoke(Request $request, $Id_GroupProvider)
    {
        $this->history->obtainOld($Id_GroupProvider);

        $this->transaction->begin();
        /** @var DownloadMediaGroupProviderResponse $response */
        $response = $this->queryBus->ask(new DownloadMediaGroupProviderQuery(
            $Id_GroupProvider,
        ));

        $this->transaction->commit();
        $index = $this->getGroupProvider($Id_GroupProvider);
        $this->history->updateBasic($Id_GroupProvider, $request);
        $groupProviderMediaList = array_map([self::class, 'appendPrivate'], $response->data()[0]['GroupProviderMedia']);
        $files = array_map(fn ($data) => $data->GroupProviderMedia_Path, $groupProviderMediaList);

        $zipFileName = $this->zipGenerator->generateFromFiles($files, $this->filename($index[0]));

        return response()->download($this->storageProvider->privatePath($zipFileName))->deleteFileAfterSend(true);
    }

    private function filename($groupProvider)
    {
        return $this->formatName(
            date('Y-m-d', strtotime($groupProvider->Group_DateStart))
                . ' '
                . $groupProvider->Service_Code
                . ' '
                . $groupProvider->Provider_NickName
        );
    }

    private function formatName(string $name)
    {
        return preg_replace('/([^ A-Za-z0-9-])/', '', $name);
    }

    private function getGroupProvider($idGroupProvider)
    {
        return DB::select(
            'CALL sp_group_provider_index(?)',
            [$idGroupProvider]
        );
    }

    public static function appendPrivate($gpm)
    {
        if ($gpm->GroupProviderMedia_Path) {
            $gpm->GroupProviderMedia_Path = config('var.PATH_PRIVATE') . $gpm->GroupProviderMedia_Path;
        }
        if ($gpm->GroupProviderMedia_Path_Thumb) {
            $gpm->GroupProviderMedia_Path_Thumb = config('var.PATH_PRIVATE') . $gpm->GroupProviderMedia_Path_Thumb;
        }
        if ($gpm->GroupProviderMedia_Path_LargeThumb) {
            $gpm->GroupProviderMedia_Path_LargeThumb = config('var.PATH_PRIVATE') . $gpm->GroupProviderMedia_Path_LargeThumb;
        }
        $gpm->GroupProviderMedia_Download = 'groupProviderMedia/' . $gpm->Id_GroupProviderMedia . '/download';
        return $gpm;
    }

    public static function appendPath($gpm)
    {
        if ($gpm->GroupProviderMedia_Path) {
            $gpm->GroupProviderMedia_Path = config('var.PATH_PUBLIC') . $gpm->GroupProviderMedia_Path;
        }
        if ($gpm->GroupProviderMedia_Path_Thumb) {
            $gpm->GroupProviderMedia_Path_Thumb = config('var.PATH_PUBLIC') . $gpm->GroupProviderMedia_Path_Thumb;
        }
        if ($gpm->GroupProviderMedia_Path_LargeThumb) {
            $gpm->GroupProviderMedia_Path_LargeThumb = config('var.PATH_PUBLIC') . $gpm->GroupProviderMedia_Path_LargeThumb;
        }
        $gpm->GroupProviderMedia_Download = 'groupProviderMedia/' . $gpm->Id_GroupProviderMedia . '/download';
        return $gpm;
    }
}
