<?php

namespace App\Http\Controllers\Owner\Group;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Modules\Group\Application\ListGroupProgramming\ListGroupProgrammingQuery;
use App\Modules\Group\Application\ListGroupProgramming\ListGroupProgrammingResponses;
use App\Modules\Group\Infraestructure\GroupValidators;
use App\Modules\Shared\Domain\Bus\Query\QueryBus;
use App\Modules\Shared\Domain\DataHandlerService;
use App\Modules\Shared\Validator\Infraestructure\LaravelValidator;
use App\Providers\ExcelGroupProgramming\ExcelGroupProgrammingReport;

class GroupProgrammingExcelListController extends Controller
{
    private $domain;
    private $provColumns;

    public function __construct(private QueryBus $queryBus)
    {
        $this->domain = "Group";
    }

    public function __invoke($action, Request $request)
    {
        (new LaravelValidator())->validate($request->all(), $this->domain, GroupValidators::PROGRAMMING_LIST);

        /** @var ListGroupProgrammingResponses $response */
        $response = $this->queryBus->ask(new ListGroupProgrammingQuery(
            $action,
            $request->get('Id_CategoryTour'),
            $request->get('Id_Tour'),
            $request->get('DateStart'),
            $request->get('DateEnd'),
            $request->header('Token', '')
        ));
        $data = (object)[
            'Tours' => $this->handleTours($response->response()),
            'ProviderColumns' => $this->provColumns,
        ];
        (new ExcelGroupProgrammingReport())->exportToExcel($data);
    }

    private function handleTours($tours)
    {
        return $this->flatGroups([$this, 'richTableHeader'])(
            $this->groupToursByDateStart(
                // $this->appendProviders(
                $this->handleProviders(
                    $this->richTours($tours)
                )
                // )
            )
        );
    }

    private function richTableHeader($tourGroup)
    {
        return (object) [
            'type' => 'header',
            'amount' => array_reduce($tourGroup->tours, fn ($sum, $tour) => $sum + $tour->Passenger_Amount, 0),
            'Passenger_Amount' => array_reduce($tourGroup->tours, fn ($sum, $tour) => $sum + $tour->Passenger_Amount, 0),
            'Group_Count' => array_reduce($tourGroup->tours, fn ($sum, $tour) => $tour->Id_Group !== null ? $sum + 1 : $sum, 0),
            'BookingTour_DateStart' => $tourGroup->element->BookingTour_DateStart,
            'childs' => json_encode($tourGroup->tours)
        ];
    }

    private function handleProviders($tours)
    {
        $typeProviderIds = array_reduce($tours, fn ($last, $g) => $this->getProviderIdsFrom($last, $g->Group_Providers), []);

        $this->provColumns = array_map(fn ($k) => (object)[
            'Id_TypeProvider' => $k,
            'title' => $typeProviderIds[$k]
        ], array_keys($typeProviderIds));
        return $tours;
    }

    private function getProviderIdsFrom($providerIds, $providers)
    {
        foreach ($providers as $prov) {
            $providerIds[$prov->Id_TypeProvider] = $prov->TypeProvider_Name;
        }
        return $providerIds;
    }

    private function flatGroups($richHeader)
    {
        $v = $this;
        return function ($dateGroups) use ($richHeader, $v) {
            return array_reduce($dateGroups, function ($data, $tourGroup) use ($richHeader, $v) {
                // $data[] = $richHeader($tourGroup);
                return array_merge($data, $v->sortGroupsByMaster($v->parseGroups($tourGroup->tours)));
            }, []);
        };
    }

    private function groupToursByDateStart($tours)
    {
        $dataHandler = new DataHandlerService();
        return $dataHandler->flatIndexGroup('tours')($dataHandler->groupDataBy($tours, 'BookingTour_DateStart'));
    }

    private function sortGroupsByMaster($groups)
    {
        $reducedGroup = array_reduce($groups, function ($prev, $curr) {
            $prev->{$curr->Id_Group} = $curr;
            return $prev;
        }, (object)[]);
        foreach ($groups as $g) {
            $g->Index = $g->Group_MasterType == 1 ? (isset($reducedGroup->{$g->Id_Master}) ? $reducedGroup->{$g->Id_Master}->Index - .5 : $g->Index) : $g->Index;
        }

        usort($groups, fn ($a, $b) => $a->Index - $b->Index);
        return $groups;
    }

    private function richTours($data)
    {
        return array_map(fn ($t) => $t, $data);
    }

    private function parseGroups($groups)
    {
        $v = $this;
        return array_map(
            function ($group, $index) use ($v) {
                $group->noValidPassengers = $v->getNoValidPassengers($group);
                $group->Index = $index;
                return $group;
            },
            $groups,
            array_keys($groups)
        );
    }

    private function getNoValidPassengers($group)
    {
        if ($group->Id_Group) {
            return $group->Group_NoValidPassengers;
        } else {
            return count(array_filter($group->Passenger_List, fn ($p) => $p->BookingTourPassenger_NoValidGroups > 0));
        }
    }
}
