<?php

namespace App\Http\Controllers\Client;

use App\Http\Controllers\Controller;
use App\Http\Helpers\ParamsValidator;
use App\Http\Helpers\ResponseBuilder;
use App\Modules\Niubiz\Domain\NiubizMdd;
use App\Providers\BookingMailSender;
use Illuminate\Support\Facades\DB;
use GuzzleHttp\Client;
use Illuminate\Http\Request;


class c_Niubiz extends Controller
{

    private $api = 'https://apiprod.vnforapps.com/';
    private $oDomain;

    public function __construct()
    {
        $this->oDomain = "Niubiz";
        if (config('var.NIUBIZ_TEST', false)) {
            $this->api = 'https://apisandbox.vnforappstest.com/';
        } else {
            $this->api = 'https://apiprod.vnforapps.com/';
        }
    }

    public function Insert(Request $request)
    {
        $oResponse  = array();

        $oResponse = ParamsValidator::Validate_Request(
            $request,
            [
                'Niubiz_Amount'                        => 'required|numeric',
                'BookingTemp_Id_User'                => 'int',
                'BookingTemp_User_Name'                => 'required|string|max:100',
                'BookingTemp_User_LastName'            => 'required|string|max:100',
                'BookingTemp_User_Email'            => 'required|string|max:250',
                'BookingTemp_User_Phone'            => 'string|max:20',
                'BookingTemp_Id_UserCountry'        => 'required|int',
                'BookingTemp_Booking'                => 'required|string',
                // 'BookingTemp_CashMovement'           => 'required|string'
            ],
            $this->oDomain
        );
        if ($oResponse['Response_Code'] != 200) {
            return ResponseBuilder::Response($oResponse);
        }

        $credentials = BookingMailSender::getCredentialList();
        $auth = BookingMailSender::filterCredential($credentials, 'Niubiz_Auth')[0];
        $merchantId = BookingMailSender::filterCredential($credentials, 'Niubiz_MerchantId')[0];


        $res = $this->getAccessToken($auth->Credential_Value);

        if (!$res['success']) {
            return $this->returnException($res['code'], 'Niubiz Access Response: ' . $res['body']);
        }
        $accessToken = $res['body'];


        $res = $this->createSessionToken(
            $accessToken,
            $request->input('Niubiz_Amount'),
            $request->ip(),
            $merchantId->Credential_Value,
            new NiubizMdd($request->input('BookingTemp_User_Email'))
        );

        if (!$res['success']) {
            return $this->returnException($res['code'], 'Niubiz Session Response: ' . $res['body']);
        }

        $oParam = [
            $request->input('BookingTemp_Id_User'),
            $request->input('BookingTemp_User_Name'),
            $request->input('BookingTemp_User_LastName'),
            $request->input('BookingTemp_User_Email'),
            $request->input('BookingTemp_User_Phone'),
            $request->input('BookingTemp_Id_UserCountry'),
            $accessToken,
            $request->input('BookingTemp_Booking'),
            $request->input('Niubiz_Amount'),
            $request->header('Token') ? $request->header('Token') : ''
            // $request->input('BookingTemp_CashMovement'),
        ];

        $oData = DB::select('call sp_client_booking_temp_insert(?,?,?,?,?,?,?,?,?,?)', $oParam);

        $oResponse["Response_Status"]            = $oData[0]->Response_Status;
        $oResponse["Response_Code"]                = $oData[0]->Response_Code;
        $oResponse["Response_Domain"]            = $this->oDomain;
        $oResponse["Response_Message"]            = $oData[0]->Response_Message;
        $oResponse["Response_Data"]                = [
            'Niubiz_AccessToken' => $accessToken,
            'Niubiz_SessionToken' => $res['body'],
            'Niubiz_MerchantId' => $merchantId,
            'Niubiz_Number' => $oData[0]->Response_Id,
            'Id_BookingTemp' => $oData[0]->Response_Id
        ];
        $oResponse["Response_Error_Message"]    = $oData[0]->Response_Message;
        $oResponse["Response_Error_Reason"]        = $oData[0]->Response_Reason;

        return ResponseBuilder::Response($oResponse);
    }

    public function createSessionToken($accessToken, $amount, $ip, $merchant, NiubizMdd $mdd)
    {
        $client = new Client();
        $body = [
            'channel' => 'web',
            'amount' => $amount,
            'antifraud' => [
                'clientIp' => $ip,
                'merchantDefineData' => [
                    'MDD4' => $mdd->mdd4Email,
                    'MDD21' => $mdd->mdd21FrequentClient,
                    'MDD32' => $mdd->mdd32ClientCode,
                    'MDD75' => $mdd->mdd75ClientType,
                    'MDD77' => $mdd->mdd77RegisterDays,
                ]
            ]
        ];
        $res = $client->request('POST', $this->api . 'api.ecommerce/v2/ecommerce/token/session/' . $merchant, [
            'headers' => [
                'Authorization' => $accessToken
            ],
            'json' => $body,
            'http_errors' => false
        ]);

        $body = '';
        switch ($res->getStatusCode()) {
            case 200:
                $body = json_decode($res->getBody()->__toString())->sessionKey;
                break;
            case 400:
                $body = json_decode($res->getBody()->__toString())->errorMessage;
                break;
            case 500:
                $body = 'Niubiz server have errors';
                break;
            default:
                $body = $res->getBody()->__toString();
        }
        return [
            'success' => $res->getStatusCode() == 200,
            'body' => $body,
            'code' => $res->getStatusCode()
        ];
    }

    public function getTransactionAuthorization($accessToken, $purchaseToken, $purchaseNumber, $amount, $merchantId)
    {

        $client = new Client();
        $res = $client->request('POST', $this->api . 'api.authorization/v3/authorization/ecommerce/' . $merchantId, [
            'headers' => [
                'Authorization' => $accessToken
            ],
            'json' => [
                'channel' => 'web',
                'captureType' => 'manual',
                'countable' => true,
                'order' => [
                    'tokenId' => $purchaseToken,
                    'purchaseNumber' => $purchaseNumber . '',
                    'amount' => $amount,
                    'currency' => 'USD'
                ]
            ],
            'http_errors' => false
        ]);

        $body = '';
        switch ($res->getStatusCode()) {
            case 200:
                $body = json_decode($res->getBody()->__toString());
                break;
            case 400:
                $body = json_decode($res->getBody()->__toString());
                break;
            case 500:
                $body = 'Niubiz server had errors';
                break;
            default:
                $body = $res->getBody()->__toString();
        }

        return [
            'success' => $res->getStatusCode() == 200,
            'body' => $body,
            'code' => $res->getStatusCode()
        ];
    }

    public function getAccessToken($auth)
    {
        $client = new Client();
        $res = $client->request('GET', $this->api . 'api.security/v1/security', [
            'headers' => [
                'Authorization' => 'Basic ' . $auth,
            ],
            'http_errors' => false
        ]);
        return [
            'success' => $res->getStatusCode() == 201,
            'body' => $res->getBody()->__toString(),
            'code' => $res->getStatusCode()
        ];
    }

    public function returnException($code, $message, $errorData = [])
    {
        $oResponse = [];
        $oResponse["Response_Status"]            = 406;
        $oResponse["Response_Code"]                = $code;
        $oResponse["Response_Domain"]            = $this->oDomain;
        $oResponse["Response_Message"]            = $message;
        $oResponse["Response_Data"]                = $errorData;
        $oResponse["Response_Error_Message"]    = $message;
        $oResponse["Response_Error_Reason"]        = $message;

        return ResponseBuilder::Response($oResponse);
    }
}
